﻿//Base class for global functions
class Base
{
    constructor()
    {
        this.MAX_STATION = 99;
        this.MAX_ADDRESS = 999999;
        this.RETRIES = 2;
        this.WATCH_DOG = 8000; //Watch dog timer in ms
        this.MIN_INT16 = -32768;
        this.MAX_INT16 = 65535;
        this.MIN_INT32 = -2147483648;
        this.MAX_INT32 = 4294967295;
        this.MIN_REAL = -3.402823466 * Math.pow(10, 38);
        this.MAX_REAL = 3.402823466 * Math.pow(10, 38);
    }

    /***************************************************************************/
    /**
     * @brief d2h - Decimal to hex converter
     *   
     * @param d: value in decimal
     ****************************************************************************/
    d2h(d)
    {
        if (d >= 0)
            return d.toString(16).toUpperCase();
        else
        {
            let threshold = 0xFFFF;
            return (threshold + d + 1).toString(16).toUpperCase();
        }
    }

    /***************************************************************************/
    /**
     * @brief d2h32 - Decimal to hex converter for 32bit values
     *   
     * @param d: value in decimal
     ****************************************************************************/
    d2h32(d)
    {
        if (d >= 0)
            return d.toString(16).toUpperCase();
        else
        {
            let threshold = 0xFFFFFFFF;
            return (threshold + d + 1).toString(16).toUpperCase();
        }
    }

    /***************************************************************************/
    /**
     * @brief h2d - Hex to decimal converter
     *   
     * @param d: value in hex
     ****************************************************************************/
    h2d(h)
    {
        return Number.parseInt(h, 16);
    }

    /***************************************************************************/
    /**
     * @brief HexStringToINT - Converts a hex value as string into a INT
     *   
     * @param value: string hex value
     * @param returns the INT value
     ****************************************************************************/
    HexStringToINT(value)
    {
        let num_base = 16;
        let negative_threshold = 0x7FFF;
        let int_val = parseInt(value, num_base);

        if (int_val > negative_threshold)
        {
            let magnitude = (int_val & negative_threshold);
            int_val = -(1 + negative_threshold - magnitude);
        }
        return int_val;
    }

    /***************************************************************************/
    /**
     * @brief HexStringToDINT - Converts a hex value as string into a DINT
     *   
     * @param value: string hex value
     * @param returns the INT value
     ****************************************************************************/
    HexStringToDINT(value)
    {
        let num_base = 16;
        let negative_threshold = 0x7FFFFFFF;
        let int_val = parseInt(value, num_base);

        if (int_val > negative_threshold)
        {
            let magnitude = (int_val & negative_threshold);
            int_val = -(1 + negative_threshold - magnitude);
        }
        return int_val;
    }

    /***************************************************************************/
    /**
     * @brief CalculateIEEEReal - calculate the real (floating number) of a given number
     * @param value: The value which should be calulated to real/float
     * @return: returns the real value
     ****************************************************************************/
    CalculateIEEEReal(value)
    {
        if (value !== 0)
        {
            let sign = value & 0x80000000 ? -1 : 1;
            let exponents = value & 0x7F800000;
            exponents = (exponents >> 23) - 127;
            let mantissa = value & 0x7FFFFF;
            return sign * (1 + (mantissa / Math.pow(2, 23))) * Math.pow(2, exponents);
        }
        else
        {
            return 0.0;
        }
    }

    /***************************************************************************/
    /**
     * @brief CalculateNumberFromIEEEReal - calculate the number of a given real/float
     * @param value: The float value which should be calulated to number
     * @return: returns the number
     ****************************************************************************/
    CalculateNumberFromIEEEReal(value)
    {
        const getHex = i => ('00' + i.toString(16)).slice(-2);
        var view = new DataView(new ArrayBuffer(4));
        view.setFloat32(0, value);
        return Number.parseInt(Array.apply(null, { length: 4 }).map((_, i) => getHex(view.getUint8(i))).join(''), 16);
    }

    /****************************************************************************
     * @brief IsBetweenAddresses -
     * Checks if the given address is between a given start and end address
     ****************************************************************************/
    IsBetweenAddresses(addrToCheck, startAddr, endAddr)
    {
        let start, end, addr;

        start = startAddr !== "0" ? `${startAddr}0` : startAddr;
        end = endAddr !== "0" ? `${endAddr}F` : "F";
        addr = addrToCheck.toString().toUpperCase();

        let startDec = 0, endDec = 0, addrInt = 0;
        for (let i = 0; i < start.length; i++)
            startDec += i !== start.length - 1 ? start.charCodeAt(i) * 100 : start.charCodeAt(i);

        for (let i = 0; i < end.length; i++)
            endDec += i !== end.length - 1 ? end.charCodeAt(i) * 100 : end.charCodeAt(i);

        for (let i = 0; i < addr.length; i++)
            addrInt += i !== addr.length - 1 ? addr.charCodeAt(i) * 100 : addr.charCodeAt(i);

        return addrInt >= startDec && addrInt <= endDec
    }

     /***************************************************************************/
    /**
     * @brief SplitValueToTwoSixteenbits - Splits a 32bit value into two 16bit values
     * @param input val: the 32bit value
     * @param output: object with the first and second 16bit value
     * @param input datatype: 0 = 16bit, 1 = 32bit, 2 = 32bit REAL etc.
     ****************************************************************************/
    SplitValueToTwoSixteenbits(val, datatype = 0)
    {
        let result = {
            first: 0,
            second: 0
        };
        let hexval;
        hexval = datatype === 0 ? this.d2h(val) : this.d2h32(val);
        let start = hexval.length >= 4 ? hexval.length - 4 : 0;
        result.first = this.HexStringToINT(hexval.substr(start));
        result.second = this.HexStringToINT(hexval.length > 4 ? hexval.substr(0, hexval.length - 4) : 0);
        return result;
    }

    /***************************************************************************/
    /**
     * @brief Utf8ArrayToStr - decode the utf8 characters into a string from the 
     * backup file
     ****************************************************************************/
    Utf8ArrayToStr(array) 
    {
        var out, i, len, c;
        var char2, char3;

        out = "";
        len = array.length;
        i = 0;
        while (i < len) 
        {
            c = array[i++];
            switch (c >> 4)
            {
                case 0: case 1: case 2: case 3: case 4: case 5: case 6: case 7:
                    // 0xxxxxxx
                    out += String.fromCharCode(c);
                    break;
                case 12: case 13:
                    // 110x xxxx   10xx xxxx
                    char2 = array[i++];
                    out += String.fromCharCode(((c & 0x1F) << 6) | (char2 & 0x3F));
                    break;
                case 14:
                    // 1110 xxxx  10xx xxxx  10xx xxxx
                    char2 = array[i++];
                    char3 = array[i++];
                    out += String.fromCharCode(((c & 0x0F) << 12) |
                        ((char2 & 0x3F) << 6) |
                        ((char3 & 0x3F) << 0));
                    break;
            }
        }

        return out;
    }

    /***************************************************************************/
    /**
     * @brief IncrementMewBoolAddress - Increment a bool address for Mewtocol
     * since Mew addresses are HEX (0, 1...F), remember only the last digit is HEX!!!
     * returns an object
     ****************************************************************************/
    IncrementMewBoolAddress(address)
    {
        let result;

        let adr = typeof (address) === "string" ? address.toUpperCase() : address.toString().toUpperCase();
        if (adr.length === 1)
        {
            if (adr !== "9" && adr !== "F")
            {
                return String.fromCharCode(adr.charCodeAt(0) + 1);
            }
            else
            {
                //After character 9 the character is ":" so we need to handle the 9 individually
                if (adr === "9")
                {
                    return "A";
                }
                //F is last address, need to add one to the word address after F
                else
                {
                    return "10";
                }
            }
        }
        //Handle address more than one character
        else
        {
            let part1, part2;
            part1 = adr.substring(0, adr.length - 1);
            part2 = adr.substring(adr.length - 1);

            if (part2 !== "9" && part2 !== "F")
            {
                return part1 + String.fromCharCode(part2.charCodeAt(0) + 1);
            }
            else
            {
                //After character 9 the character is ":" so we need to handle the 9 individually
                if (part2 === "9")
                {
                    return part1 + "A";
                }
                //F is last address. Now we need to check if the part1 also have addresses with F, if so they all must be incremented
                else
                {
                    part2 = "0";
                    let i, zeroes = "";
                    for (i = part1.length - 1; i >= -1; i--)
                    {
                        if (part1[i] === "9")
                        {
                            zeroes += "0";
                        }
                        else
                        {
                            break;
                        }
                    }

                    //Seems like all others are 9, add a 1 to the word address
                    if (i < 0)
                    {
                        result = "1" + zeroes + part2;
                    }
                    else
                    {
                        if (part1[i] !== "9")
                        {
                            part1 = part1.slice(0, i) + String.fromCharCode(part1.charCodeAt(i) + 1) + zeroes;
                            result = part1 + part2;
                        }
                        else
                        {
                            part1 = part1.slice(0, i) + "0" + zeroes;
                            result = part1 + part2;
                        }
                    }

                    return result;
                }
            }
        }
    }

    /***************************************************************************/
    /**
     * @brief AddToMewBoolAddress - calculate the new mewtocol bool address with an offset
     * e.g. R19F + 10 -> R209
     * returns an object
     ****************************************************************************/
    AddOffsetToMewBoolAddress(address, offset)
    {
        let words, bits, sum;
        //Address one digit long - word address = bit address
        if (address.length === 1)
        {
            sum = parseInt(address, 16) + offset;
            words = Math.floor(sum / 16);
            bits = (sum % 16).toString(16).toUpperCase();

            if (words > 0)
                return words.toString() + bits;

            return bits;
        }
        else
        {
            //Address more than 1 digit - word and bit address separated
            let wordAddr = parseInt(address.substring(0, address.length - 1));
            let bitAddr = address.substring(address.length - 1);

            sum = parseInt(bitAddr, 16) + offset;
            words = Math.floor(sum / 16);
            bits = (sum % 16).toString(16).toUpperCase();

            wordAddr += words;

            return wordAddr.toString() + bits;
        }
    }

    /***************************************************************************
     * @brief bcc - calculate the bcc (block check code) of a ascii command
     ****************************************************************************/
    bcc(cmd)
    {
        let currentval;
        let previousVal = 0;
        let result;

        for (let i = 0; i < cmd.length; i++)
        {
            currentval = Number(cmd.charCodeAt(i));
            result = previousVal ^ currentval;
            previousVal = result;
        }

        if (result.toString(16).length === 1)
            return `0${result.toString(16).toUpperCase()}`

        return result.toString(16).toUpperCase();
    }

    /***************************************************************************
     * @brief byteSwap - swap the 2 bytes of 16bit value
     ****************************************************************************/
    byteSwap(val)
    {
        return ((val & 0xFF) << 8) | ((val >> 8) & 0xFF);
    }

    /***************************************************************************
     * @brief ip2int - converts an ip as string into a number
     ****************************************************************************/
    ip2int(ip) {
        return (
            ip.split(".").reduce(function (ipInt, octet) {
                return (ipInt << 8) + parseInt(octet, 10);
            }, 0) >>> 0
        );
    }

    /***************************************************************************
     * @brief int2ip - converts a number into an ip as string
     ****************************************************************************/
    int2ip(val) {
        return (
            (val >>> 24) +
            "." +
            ((val >> 16) & 255) +
            "." +
            ((val >> 8) & 255) +
            "." +
            (val & 255)
        );
    }

    /***************************************************************************
     * @brief isUndefined - returns true if typeof value is undefined
     ****************************************************************************/
    isUndefined(value) {
        return typeof value === "undefined";
    }

    /***************************************************************************
     * @brief isStationNumberValid - returns true if station number is valid
     ****************************************************************************/
    isStationNumberValid(station) {
        return station >= 0 && station <= this.MAX_STATION;
    }

    /***************************************************************************
     * @brief getStationNumberAsValidString - returns the station number as string
     * for Mewtocol commands
     ****************************************************************************/
    getStationNumberAsValidStringMew(station) {
        switch (true) {
            case station === 0:
                return "EE";
            case station >= 1 && station < 10:
                return `0${station.toString()}`;
            case station >= 10:
                return station.toString();

            default:
                return "EE";
        }
    }
    /***************************************************************************
     * @brief getStationNumberAsValidString - returns the station number as string
     * for Mewtocol7 commands
     ****************************************************************************/
    getStationNumberAsValidStringMew7(station) {
        switch (true) {
            case station === 0:
                return "EEE";
            case station >= 1 && station < 10:
                return `00${station.toString()}`;
            case station >= 10 && station < 100:
                return `0${station.toString()}`;
            case station >= 100:
                return station.toString();

            default:
                return "EEE";
        }
    }
}

module.exports = Base;